//*************************************************************************************
// file 		: SQ7515_SIO.c
// version		: V1.0
// brief		: SIO related functions
// note			: SIO related functions are gathered in this subroutine
//**************************************************************************************
#include "main.h"
unsigned char tmpChar, s_rx;


//*********************************************************************
// name			: SIO0_init()
// brief		: SIO0 initialization
// note			: set SIO0 pins and transfer mode
//			  
//*********************************************************************
void SIO0_Init(){
/*	
    //--- Master mode (SIOCKS = 000 to 110)-------------------
    //====== SIO0 pin configuration initialization ======

    //--- SI0(P0.5) ; SO0(P0.4) ; SCK0(0.6)-------
    FSELR0 = (FSELR0&0x0F)|0x20;        // set function selection register 0
    PCSELR2_SIO0 = 0x00;                // set peripheral channel selection register

    P0FC1_P5 = 1;                       // set P0.5 to SI0
    P0FC2_P5 = 0;
    P0FC1_P4 = 1;                       // set P0.4 to SO0
    P0FC2_P4 = 0;	
    P0FC1_P6 = 1;                       // set P0.6 to SCK0
    P0FC2_P6 = 0;
*/


/*
    //--- SI0(P2.5) ; SO0(P2.6) ; SCK0(2.4)-------
    FSELR2 = (FSELR2&0x0F)|0x20;        // set function selection register2
    PCSELR2_SIO0 = 0x01;                // set peripheral channel selection register

    P2FC1_P5 = 1;                       // set P2.5 to SI0
    P2FC2_P5 = 0;
    P2FC1_P6 = 1;                       // set P2.6 to SO0
    P2FC2_P6 = 0;	
    P2FC1_P4 = 1;                       // set P2.4 to SCK0
    P2FC2_P4 = 0;
*/


/*
    PCKEN2_SIO0 = 1;                    // enable SIO0
    SIO0CR1_SIOEDG  = 0;                // set transfer edge to receive data at a rising edge and transmit data at a falling edge
    SIO0CR1_SIOCKS  = 0x00;             // set serial clock to fcgck/2^9
    SIO0CR1_SIODIR  = 1;                // set transfer format to MSB first
    SIO0CR1_SIOM    = 0x03;             // set transfer mode to 8-bit transmit and receive mode
*/

}



//*********************************************************************
// name			: SIO1_init()
// brief		: SIO1 initialization
// note			: set SIO1 pins and transfer mode
//			  *set SIO1 to Master mode in this example
//*********************************************************************
void SIO1_Init(){

    //--- Master mode (SIOCKS = 000 to 110)-------------------
    //====== SIO1 pin configuration initialization ======
    //--- SI1(P0.0) ; SO1(P0.1) ; SCK1(0.2)-------[this pin is selected for this example]
    FSELR0 = (FSELR0&0xF0)|0x02;        // set function selection register 0
    PCSELR2_SIO1 = 0x00;                // set peripheral channel selection register

    P0FC1_P0 = 1;                       // set P0.0 to SI1
    P0FC2_P0 = 0;
    P0FC1_P1 = 1;                       // set P0.1 to SO1
    P0FC2_P1 = 0;	
    P0FC1_P2 = 1;                       // set P0.2 to SCK1
    P0FC2_P2 = 0;
	
/*
    //--- SI1(P2.1) ; SO1(P2.0) ; SCK1(2.2)-------
    FSELR2 = (FSELR2&0xF0)|0x02;        // set function selection register2
    PCSELR2_SIO1 = 0x01;                // set peripheral channel selection register

    P2FC1_P1 = 1;                       // set P2.1 to SI1
    P2FC2_P1 = 0;
    P2FC1_P0 = 1;                       // set P2.0 to SO1
    P2FC2_P0 = 0;	
    P2FC1_P2 = 1;                       // set P2.2 to SCK1
    P2FC2_P2 = 0;
*/

    PCKEN2_SIO1 = 1;                    // enable SIO1

    SIO1CR1_SIOEDG  = 0;                // set transfer edge to receive data at a rising edge and transmit data at a falling edge
    //SIO1CR1_SIOCKS  = 0x00;            // set serial clock to fcgck/2^9
    SIO1CR1_SIOCKS  = SIO_CKS_DIV16;    // set serial clock to fcgck/16
	
    SIO1CR1_SIODIR  = 1;                // set transfer format to MSB first
    SIO1CR1_SIOM    = 0x03;             // set transfer mode to 8-bit transmit and receive mode
    //SIO1CR1 = 0x0B|(SIO_CKS_DIV512<<4);
	
}



//*********************************************************************
// name			: SIO_Start()
// brief		: start SIO
// note			:
//*********************************************************************
void SIO_Start(unsigned char sio){

	P4DO_P3 = 0;							  // simulate SS pin in SPI interface to output L
	if(sio == SIO_0){
		SIO0CR1_SIOS  = 1;               	  // SIO0 starts operation
		SIO0CR2_SPIEN = 1;              	  // enable SIO0
	}
	else if(sio == SIO_1){
		SIO1CR1_SIOS  = 1;               	  // SIO1 starts operation 
		SIO1CR2_SPIEN = 1;              	  // enable SIO1
	}

}


//*********************************************************************
// name			: SIO0_Stop()
// brief		: stop SIO0
// note			:
//*********************************************************************
void SIO_Stop(unsigned char sio){

	if(sio == SIO_0){
		SIO0CR1_SIOS  = 0;               	  // SIO0 stops operation 
		SIO0CR2_SPIEN = 0;					  // disable SIO0
	}
	else if(sio == SIO_1){
		SIO1CR1_SIOS  = 0;               	  // SIO1 stops operation 
		SIO1CR2_SPIEN = 0;					  // disable SIO1
	}
	P4DO_P3 = 1;							  // simulate SS pin in SPI interface to output H
}


//*********************************************************************
// name			: SIO_IntEnable()
// brief		: enable SIO interrupt
// note			:
//*********************************************************************
void SIO_IntEnable(unsigned char sio){

    __ASM("DI");                        // enable all maskable interrupts
	
    if(sio == SIO_0){
        IER_SIO0 = 1;                   // enable SIO0 interrupt
    }
    else if(sio == SIO_1){
        IER_SIO1 = 1;                   // enable SIO1 interrupt
    }

	__ASM("EI");                        // enable all maskable interrupts
}



//*********************************************************************
// name			: SIO_ByteSend()
// brief		: SO transmits data
// note			: 
//*********************************************************************
void SIO_ByteSend(unsigned char sio, unsigned char txdata){

	if(sio == SIO_0){  
		while(SIO0SR_TBFL == 1);      			// wait for transmit buffer to be empty
		SIO0BUF = txdata;           			// put data to transmit buffer
		//__ASM("NOP");							// no operation
		//  CLR_WDT;                			// clear the watchdog timer 
        
		SIO0CR1_SIOS = 1;       				// start SIO
		while(SIO0SR_SEF == 1);           		// wait for shift operation not in progress   
		tmpChar = SIO0BUF;						// read data from slave	
	}
	else if(sio == SIO_1){
		while(SIO1SR_TBFL == 1);      			// wait for transmit buffer to be empty
		SIO1BUF = txdata;           			// put data to transmit buffer
		//__ASM("NOP");							// no operation
		//  CLR_WDT;                			// clear the watchdog timer 
        
		SIO1CR1_SIOS = 1;       				// start SIO
		while(SIO1SR_SEF == 1);           		// wait for shift operation not in progress   
		tmpChar = SIO1BUF;						// read data from slave	
	}
}


//*********************************************************************
// name			: SIO_ByteRead()
// brief		: SI receives data
// note			:
//*********************************************************************
unsigned char SIO_ByteRead(unsigned char sio){

    unsigned char r_data;

	if(sio == SIO_0){
		while(SIO0SR_TBFL == 1);	// wait for transmit buffer to be empty
		SIO0BUF = 0xff;				// put dummy data to buffer to generate clock for slave
		tmpChar = SIO0BUF;			// read data from slave
		while(SIO0SR_RENDB == 0);	// wait for no more received data since the last receive data was read out
		r_data = SIO0BUF;		    // read data
	}
	else if(sio == SIO_1){
		while(SIO1SR_TBFL == 1);	// wait for transmit buffer to be empty
		SIO1BUF = 0xff;				// put dummy data to buffer to generate clock for slave
		tmpChar = SIO1BUF;			// read data from slave
		while(SIO1SR_RENDB == 0);	// wait for no more received data since the last receive data was read out
		r_data = SIO1BUF;		    // read data
	}
	return r_data;
}


//*********************************************************************
// name			: SIO_ReservedStopWait
// brief		: wait for SIO stop
// note			: make sure transfer not in progress
//*********************************************************************
void SIO_ReservedStopWait(unsigned char sio){

    if(sio == SIO_0){
        SIO0CR1_SIOS = 0;               // SIOS=0: reserved stop

        while( SIO0SR_SIOF == 1 ){      // SIOF: wait for SIO stop(SIOF=0)    
            //__ASM("NOP");               // no operation
        }
    }
    else  if(sio == SIO_1){
        SIO1CR1_SIOS = 0;               // SIOS=0: reserved stop

        while( SIO1SR_SIOF == 1 ){      // SIOF: wait for SIO stop(SIOF=0)    
            //__ASM("NOP");               // no operation
        }
    }

}


//*********************************************************************
// name			: __interrupt IntSIO0(void)
// brief		: SIO0 interrupt service routine
// note			: 
//*********************************************************************
void __interrupt IntSIO0(void){	
    __ASM("NOP");                       // no operation
}

//*********************************************************************
// name			: __interrupt IntSIO1(void)
// brief		: SIO1 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntSIO1(void){
    __ASM("NOP");                       // no operation
}

/* ------------------------- end of file ---------------------------- */
