//*************************************************************************************
// file			: SQ7515_SecureUnit_I2C.c
// version		: V1.0
// brief		: the I2C functions that controls SecureUnit
// note			:
//**************************************************************************************
#include "main.h"


//*********************************************************************
// name			: I2C0_Init()
// brief		: I2C0 initialization
// param		: 
// retval		: 
// note			: I2C0 setting, 
//			  *this example sets I2C0 to Slave mode , Slave address = 0x68 
//*********************************************************************
void I2C0_Init(){

	
//====== I2C0pin configuration initialization ======
	//--SDA0 (P0.5) ; SCL0 (P0.6)------- [this pin is selected for this example]
	P0FC1_P5 = 1;                       // set P0.5 as SDA0
	P0FC2_P5 = 0;
	P0FC1_P6 = 1;                       // set P0.6 as SCL0
	P0FC2_P6 = 0;
	FSELR0 |= 0x30;						// set function selection register 0
	PCSELR1_I2C0 = 0x00;				// select peripheral function channel

/*	
	//--SDA0 (P0.4) ; SCL0 (P0.5)-------
	P0FC1_P4 = 1;                       // set P0.4 as SDA0
	P0FC2_P4 = 0;
	P0FC1_P5 = 1;                       // set P0.5 as SCL0
	P0FC2_P5 = 0;
	FSELR0 |= 0x40;						// set function selection register 0
	PCSELR1_I2C0 = 0x01;				// select peripheral function channel
*/	


/*
	//--SDA0 (P2.5) ; SCL0 (P2.4)-------
	P2FC1_P5 = 1;                       // set P2.5 as SDA0
	P2FC2_P5 = 0;
	P2FC1_P4 = 1;                       // set P2.4 as SCL0
	P2FC2_P4 = 0;
	FSELR2 |= 0x30;						// set function selection register 2
	PCSELR1_I2C0 = 0x02;				// select peripheral function channel
*/	

	
/*	
	//--SDA0 (P3.2) ; SCL0 (P3.3)-------
	P3FC1_P2 = 1;                       // set P3.2 as SDA0
	P3FC2_P2 = 0;
	P3FC1_P3 = 1;                       // set P3.3 as SCL0
	P3FC2_P3 = 0;
	FSELR3 |= 0x00;						// set function selection register 3
	PCSELR1_I2C0 = 0x03;				// select peripheral function channel
*/


//====== I2C0 setting ======
	PCKEN2_I2C0 = 1;                    // enable I2C0
	SBI0CR1_SCK = SCK_HI_135_LO_138;    // set HIGH and LOW periods of the serial clock in the master mode Time before the release of the SCL pin in the slave mode
	
	SBI0CR1_ACK = 1;                    // acknowledgement mode
	SBI0CR1_BC = 0x00;                  // set number of clocks for data transfer to 9 bits


	
//--- set I2C0 to Slave mode ---				*I2C0 is Slave mode in this example
	//------------bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ----
	//--- SBI0CR2 <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ----
	//--- <MST>   1 : Master
	//            0 : Slave
	//--- <TRX>   1 : transmitter   ; 0: receiver
	//--- <BB>    1 : generate the start condition
	//            0 : generate the stop condition
	//--- <PIN>	  1 : cancel interrupt service request
	//
	//--- <SBIM>  1 : Serial bus interface mode / 0:Port mode
	//--- <SWRST> software reset start bit  
	//------------------------------------------------------------------------
	

	//---if I2C0 is Slave mode, Slave address must be set
	I2C0AR = SLAVE_ADDR << 1;			// set Slave address
	
	//---if it is Slave mode, SBICR2<PIN> and <SBIM> must be set
	SBI0CR2 = 0x18;						// Slave mode 0001 1000
	

	
	
}


//*********************************************************************
// name			: I2C0_IntEnable()
// brief		: enable I2C0 interrupt
// param		: 
// retval		: 
// note			: 
//*********************************************************************
void I2C0_IntEnable(void)
{
	__ASM("DI");						// enable all maskable interrupts
	IFR_I2C0 = 0;                       // clear interrupt flag
	IER_I2C0 = 1;	                    // enable I2C0 interrupt
	__ASM("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: I2C0_IntDisable() 
// brief		: disable I2C0 interrupt
// param		: 
// retval		: 
// note			: 
//*********************************************************************
void I2C0_IntDisable(void)
{
	__ASM("DI");						// enable all maskable interrupts
	IFR_I2C0 = 0;                       // clear interrupt flag
	IER_I2C0 = 0;	                    // disable I2C0 interrupt 
	__ASM("EI");						// enable all maskable interrupts
}



//*********************************************************************
// name			: I2C1_Init()
// brief		: I2C1 initialization
// param		: 
// retval		: 
// note			: I2C1 setting,  
//			  *this example sets I2C1 to Master mode
//*********************************************************************
void I2C1_Init(){
	
	
//====== I2C1pin configuration initialization ======

/*
	//--SDA1 (P0.0) ; SCL1 (P0.2)-------
	P0FC1_P0 = 1;                       // set P0.0 as SDA1
	P0FC2_P0 = 0;
	P0FC1_P2 = 1;                       // set P0.2 as SCL1
	P0FC2_P2 = 0;
	FSELR0 |= 0x03;						// set function selection register 0
	PCSELR1_I2C1 = 0x00;				// select peripheral function channel
*/

	//--SDA1 (P2.1) ; SCL1 (P2.2)-------
	P2FC1_P1 = 1;                       // set P2.1 as SDA1
	P2FC2_P1 = 0;
	P2FC1_P2 = 1;                       // set P2.2 as SCL1
	P2FC2_P2 = 0;
	FSELR2 |= 0x00;						// set function selection register 2
	PCSELR1_I2C1 = 0x01;				// select peripheral function channel


/*

	//--SDA1 (P4.6) ; SCL1 (P4.7)-------  [this pin is selected for this example]
	P4FC1_P6 = 1;                       // set P4.6 as SDA1
	P4FC2_P6 = 0;
	P4FC1_P7 = 1;                       // set P4.7 as SCL1
	P4FC2_P7 = 0;
	FSELR4 |= 0x30;						// set function selection register 4
	PCSELR1_I2C1 = 0x02;				// select peripheral function channel
*/

//====== I2C1 setting ======
	PCKEN2_I2C1 = 1;                    // enable I2C1
	SBI1CR1_SCK = SCK_HI_39_LO_42;    // set HIGH and LOW periods of the serial clock in the master mode Time before the release of the SCL pin in the slave mode
	
	SBI1CR1_ACK = 1;                    // acknowledgement mode
	SBI1CR1_BC = 0x00;                  // set number of clocks for data transfer to 9 bits	
	

//--- [set I2C1 to Master mode] ---           * I2C1 is Master mode in this example
//------------bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ----
//--- SBI1CR2 <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ----
//--- <MST>   1 : Master
//            0 : Slave
//--- <TRX>   1 : transmitter   ; 0: receiver
//--- <BB>    1 : generate the start condition
//            0 : generate the stop condition
//--- <PIN>	  1 : cancel interrupt service request
//
//--- <SBIM>  1 : Serial bus interface mode / 0:Port mode
//--- <SWRST> software reset start bit  
//------------------------------------------------------------------------	
	//---if it is Master mode, SBICR2<MST>, <PIN> and <SBIM> must be set
	SBI1CR2 = 0xC8;						// Master mode 1100 1000
	//SBI1CR2 = SBICR2_MST|SBICR2_PIN|SBICR2_SBIM;  // Master mode
	IFR_I2C1 = 0;                       // clear interupt flag                      
	IER_I2C1 = 0;                       // disable I2C1 interrupt

	
}


//*********************************************************************
// name			: I2C1_IntEnable()
// brief		: enable I2C1 interrupt
// param		: 
// retval		: 
// note			: 
//*********************************************************************
void I2C1_IntEnable(void)
{
	__ASM("DI");						// enable all maskable interrupts
	IFR_I2C1 = 0;                       // clear interrupt flag
	IER_I2C1 = 1;	                    // enable I2C1 interrupt
	__ASM("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: I2C1_IntDisable() 
// brief		: disable I2C1 interrupt
// param		: 
// retval		: 
// note			: 
//*********************************************************************
void I2C1_IntDisable(void)
{
	__ASM("DI");						// enable all maskable interrupts
	IFR_I2C1 = 0;                       // clear interrupt flag
	IER_I2C1 = 0;	                    // disable I2C1 interrupt 
	__ASM("EI");						// enable all maskable interrupts	
}


//*********************************************************************
// name			: I2C_Start() [Master mode]
// brief		: generate a start conditon
// param		: i2c: I2C_0, I2C_1       I2C selection 
//            adr: slave address
// retval		: 
// note			: 
//*********************************************************************
//SBI0CR2/SBI1CR2
//-----bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ---------
//---  <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ---------
//--- <MST>   1 : Master
//            0 : Slave
//--- <TRX>   1 : transmitter   ; 0: receiver
//--- <BB>    1 : generate the start condition
//            0 : generate the stop condition
//--- <PIN>	  1 : cancel interrupt service request
//
//--- <SBIM>  1 : Serial bus interface mode / 0:Port mode
//--- <SWRST> software reset start bit  
//----------------------------------------------------------------------
void I2C_Start(unsigned char i2c,unsigned char adr){
	
	if(i2c==I2C_0){
		unsigned char tWait_I2C0 = 0;		// when there is no response, the initial value of the wait
		while(SBI0SR_BB);					// SBI0SR_BB: Bus status confirms is free ; 1 :busy  0: free

// 		SBI0CR1 = 0x10;						// ***for each start conditon, set SBI0CR1
// 		//****** Write data to the registers before the start condition is generated ******	
// 		
// 	
 		SBI0DBR = adr;						// set slave device address
// 		SBI0CR2 = 0xf8;						// Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
// 											// 1111 1000
// 											// setting : Master, Transmitter, generate the start condition, cancel interrupt service request and master mode
		SBI0CR2 = SBICR2_MST|SBICR2_TX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
		
 		while(SBI0SR_PIN){};             	// wait for cancel interrupt service request
 		
 		
// 		// ---check the last received bit, check <LRB> slave answer signal ACK---
// 		while(SBI0SR_LRB = 1){					
// 				tWait_I2C0++;				// keep waiting
// 				if( tWait_I2C0>250 ){		// if there is no response for too long, the loop will be broken		
// 					I2C_Stop(i2c);				// I2C stop -> release SCK
// 					break;
// 				}	
// 		}
		
	}	
	else if(i2c==I2C_1){
		unsigned char tWait_I2C1 = 0;		// when there is no response, the initial value of the wait
		while(SBI1SR_BB);					// SBI1SR_BB: Bus status confirms is free ; 1 :busy  0: free

// 		SBI1CR1 = 0x10;						// ***for each start conditon, set SBI1CR1
// 		//****** Write data to the registers before the start condition is generated ******	
// 	
 		SBI1DBR = adr;						// set slave device address
// 		SBI1CR2 = 0xf8;						// Write "1" to SBI1CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
// 											// 1111 1000
// 											// setting : Master, Transmitter, generate the start condition, cancel interrupt service request and master mode
		
		SBI1CR2 = SBICR2_MST|SBICR2_TX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;

 		while(SBI1CR2_PIN){};             	// wait for cancel interrupt service request
 		
// 		// ---check the last received bit, check <LRB> slave answer signal ACK---
// 		while(SBI1SR_LRB = 1){					
// 				tWait_I2C1++;				// keep waiting
// 				if( tWait_I2C1>250 ){		// if there is no response for too long, the loop will be broken		
// 					I2C_Stop(i2c);		    // I2C stop -> release SCK
// 					break;
// 				}	
// 		}
// 

 	}
	
}

//*********************************************************************
// name			: I2C_ReStart() [Master mode]
// brief		: transfer operation ReStart instruction
// param		: i2c: I2C_0, I2C_1       I2C selection 
//            adr: slave address
// retval		: 
// note			: 
//*********************************************************************
void I2C_ReStart(unsigned char i2c,unsigned char adr){

	if(i2c==I2C_0){
// 		SBI0CR2 = 0x18;						// Write "1" to SBI0CR2<PIN>  and <SBIM>to "1"
// 											// 00001 1000
// 											// setting : cancel interrupt service request and serial bus interface mode
		SBI0CR2 = SBICR2_PIN|SBICR2_SBIM;
		while(SBI0SR_BB == 1);				// SBI0SR_BB: Bus status confirms is free ; 1 :busy  0: free	
		while(SBI0SR_LRB == 0);				// wait <LRB>=1
	}
	else if(i2c==I2C_1){
// 		SBI1CR2 = 0x18;						// Write "1" to SBI1CR2<PIN>  and <SBIM>to "1"
// 											// 00001 1000
// 											// setting : cancel interrupt service request and serial bus interface mode
		SBI1CR2 = SBICR2_PIN|SBICR2_SBIM;
		while(SBI1SR_BB == 1);				// SBI1SR_BB: Bus status confirms is free ; 1 :busy  0: free	
		while(SBI1SR_LRB == 0);				// wait <LRB>=1
	}

	I2C_Start(i2c,adr);                 	// generate a start conditon again

}


//*********************************************************************
// name			: I2C_Stop() [Master mode]
// brief		: transfer operation Stop instruction
// param		: i2c: I2C_0, I2C_1       I2C selection 
// retval		: 
// note			: 
//*********************************************************************
//SBI0CR2/SBI1CR2
//-----bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ---------
//---        <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ---
//--- <MST>   1 : Master
//            0 : Slave
//--- <TRX>   1 : transmitter   ; 0: receiver
//--- <BB>    1 : generate the start condition
//            0 : generate the stop condition
//--- <PIN>	  1 : cancel interrupt service request
//
//--- <SBIM>  1 : Serial bus interface mode / 0:Port mode
//--- <SWRST> software reset start bit  
//----------------------------------------------------------------------
void I2C_Stop(unsigned char i2c){
	if(i2c==I2C_0){
// 	    SBI0CR2 = 0xd8;                 // Set SBI0CR2<MST>, <TRX> and <PIN> to "1" and SBI0CR2<BB> to "0"
//                                      // 1101 1000
//                                      // setting : Master, Transmitter, generate the stop condition, cancel interrupt service request and port  mode
        SBI0CR2 = SBICR2_MST|SBICR2_TX|SBICR2_PIN|SBICR2_SBIM;
	
        while(SBI0SR_BB);               // wait for I2C Bus free,SBI0SR_BB: Bus status confirms is free ; 1 :busy  0: free
    }
	else if(i2c==I2C_1){
// 		SBI0CR2 = 0xd8;                 // Set SBI0CR2<MST>, <TRX> and <PIN> to "1" and SBI0CR2<BB> to "0"
//                                      // 1101 1000
//                                      // setting : Master, Transmitter, generate the stop condition, cancel interrupt service request and port  mode
        SBI1CR2 = SBICR2_MST|SBICR2_TX|SBICR2_PIN|SBICR2_SBIM;
        while(SBI1SR_BB);               // wait for I2C Bus free,SBI1SR_BB: Bus status confirms is free ; 1 :busy  0: free
	}
}

//*********************************************************************
// name			: I2C_TXOneByte() [Master mode]
// brief		: transfer a single byte
// param		: i2c: I2C_0, I2C_1       I2C selection 
//            TxBuf: data to be tranferred
// retval		: 
// note			: 
//*********************************************************************
void I2C_TXOneByte(unsigned char i2c,unsigned char TxBuf){
	
	if(i2c==I2C_0){
		SBI0DBR = TxBuf;
		SBI0CR2 = SBICR2_MST|SBICR2_TX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
		while(SBI0CR2_PIN);             // wait for TX completed
	}
	else if(i2c==I2C_1){
		SBI1DBR = TxBuf;
		SBI1CR2 = SBICR2_MST|SBICR2_TX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
		while(SBI1CR2_PIN);             // wait for TX completed
	}
}



//*********************************************************************
// name			: I2C_RXOneByte() [Master mode]
// brief		: receive a single byte
// param		: i2c: I2C_0, I2C_1       I2C selection 
// retval		: received data
// note			: 
//*********************************************************************
unsigned char I2C_RXOneByte(unsigned char i2c){
	if(i2c==I2C_0)
	{
		SBI0CR2 = SBICR2_MST|SBICR2_RX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
		SBI0DBR = 0x00;                 // clear buffer
		while(SBI0CR2_PIN == 1);        
		return SBI0DBR;
	}
	else if(i2c==I2C_1)
	{
		SBI1CR2 = SBICR2_MST|SBICR2_RX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
		SBI1DBR = 0x00;                 // clear buffer
		while(SBI1CR2_PIN == 1);
		return SBI1DBR;
	}
	else
		return 0;
}


//*********************************************************************
// name			: IsAck()
// brief		: check ACK
// param		: i2c: I2C_0, I2C_1       I2C selection 
// retval		: 1: NACK   0: ACK
// note			: 
//*********************************************************************
unsigned char IsAck(unsigned char i2c){

	if(i2c==I2C_0){
		return SBI0SR_LRB;
	}
	else if(i2c==I2C_1){
		return SBI1SR_LRB;
	}

	return I2C_NACK;
}


//*********************************************************************
// name			: I2C_AckEnable()
// brief		: set acknowledgement mode or non-acknowledgement mode
// param		: i2c: I2C_0, I2C_1       I2C selection 
//            enable: 1 acknowledgement mode    0 non-acknowledgement mode
// retval		:
// note			: 
//*********************************************************************
void I2C_AckEnable(unsigned char i2c, unsigned char enable){
	if(i2c==I2C_0){
		SBI0CR1_ACK = enable;				
	}
	else if(i2c==I2C_1){
		SBI1CR1_ACK = enable;				
	}
}


//*********************************************************************
// name			: I2C_BufCount()
// brief		: set number of data bits
// param		: i2c: I2C_0, I2C_1       I2C selection 
//            bufcount: 
//                   BC_8bit      0x00
//                   BC_1bit      0x01
//                   BC_2bit      0x02
//                   BC_3bit      0x03
//                   BC_4bit      0x04
//                   BC_5bit      0x05
//                   BC_6bit      0x06
//                   BC_7bit      0x07 
// note			:
//*********************************************************************
void I2C_BufCount(unsigned char i2c, unsigned char bufcount){

	if(i2c==I2C_0){	
		SBI0CR1_BC = bufcount;
	}
	else if(i2c==I2C_1){	
		SBI1CR1_BC = bufcount;
	}

}


//*********************************************************************
// name			: I2C_ReadState()
// brief		: read I2C status
// param		: i2c: I2C_0, I2C_1       I2C selection 
// retval		:
// note			:
//*********************************************************************
unsigned char I2C_ReadState(unsigned char i2c){

	if(i2c==I2C_0){
		return SBI0SR;						// read I2C0 status
	}
	else if(i2c==I2C_1){
		return SBI1SR;						// read I2C1 status
	}
	else{
		return 0;
    }

}


//*********************************************************************
// name			: I2C_SetBufvalue()
// brief		: write data to I2C buffer
// param		: i2c: I2C_0, I2C_1       I2C selection 
//            data:  data to be transmitted
// retval		:
// note			:
//*********************************************************************
void I2C_SetBufvalue(unsigned char i2c,unsigned char data){

	if(i2c==I2C_0){
		SBI0DBR = data;
	}
	else if(i2c==I2C_1){
		SBI1DBR = data;
	}

}

//*********************************************************************
// name			: I2C_ReadBufvalue()
// brief		: read out I2C buffer data
// param		: i2c: I2C_0, I2C_1       I2C selection 
// retval		: I2C buffer data
// note			: 
//*********************************************************************
unsigned char I2C_ReadBufvalue(unsigned char i2c){

	if(i2c==I2C_0){
        return SBI0DBR;
	}
	else if(i2c==I2C_1){
		return SBI1DBR;
	}
	else{
		return 0;
	}
}


//*********************************************************************
// name			: I2C_FinishSlaveTx() [Slave mode]
// brief		: I2C Slave transfer completion to release the I2C bus
// param		: i2c: I2C_0, I2C_1       I2C selection 
// retval		:
// note			: 
//*********************************************************************
void I2C_FinishSlaveTx(unsigned char i2c){

	if(i2c==I2C_0){
		SBI0CR2 = 0x58;
		SBI0CR2 = 0x18;
	}
	else if(i2c==I2C_1){
		SBI1CR2 = 0x58;
		SBI1CR2 = 0x18;
	}

}


//*********************************************************************
// name			: I2C_Reset()
// brief		: I2C software reset. SBI0CR1,I2C0AR,SBI0CR2 reset,SBI0CR2<SBIM> not reset
// param		: i2c: I2C_0, I2C_1       I2C selection 
// retval		:
// note			: 
//*********************************************************************
void I2C_Reset(unsigned char i2c){

	if(i2c==I2C_0){
		SBI0CR2 = 0x1A;
		SBI0CR2 = 0x19;
	}
	else if(i2c==I2C_1){
		SBI1CR2 = 0x1A;
		SBI1CR2 = 0x19;
	}

}


//*********************************************************************
// name			: __interrupt IntI2C0()
// brief		: I2C0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntI2C0(void)
{
	__asm("NOP");						// no operation

	//I2C_Slave_Int(I2C_0,&i2c_slave);
}


//*********************************************************************
// name			: __interrupt IntI2C1()
// brief		: I2C1 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntI2C1(void)
{
	__asm("NOP");						// no operation

	
}


//*********************************************************************
// name			: I2C_Master_Write()
// brief		: 
// param		: i2c: I2C_0, I2C_1       I2C selection 
//            pi2c: packet data pointer
// retval		: 1: successful   0: failed
// note			:
//*********************************************************************
unsigned char I2C_Master_Write(unsigned char i2c,I2C_MasterPacket* pi2c){
	unsigned char cnt;
	
	I2C_Start(i2c,(unsigned char)(pi2c->adr)|I2C_Write);

	if(IsAck(i2c)==I2C_NACK){
		I2C_Stop(i2c);
		return 0;
	}		
	
	for(cnt=0; cnt<pi2c->len; cnt++){
		I2C_TXOneByte(i2c,*pi2c->pBuf++);  // send one byte
		if(IsAck(i2c)==I2C_NACK){
		    I2C_Stop(i2c);
		    return 0;
	    }	
	}

	I2C_Stop(i2c);

	return 1;
}



//*********************************************************************
// name			: I2C_Master_Read()
// brief		: 
// param		: i2c: I2C_0, I2C_1       I2C selection 
//            pi2c: packet data pointer
// retval		: 1: successful   0: failed
// note			:
//*********************************************************************
unsigned char I2C_Master_Read(unsigned char i2c,I2C_MasterPacket* pi2c)
{
	unsigned char cnt,length;
	
	
	I2C_Start(i2c,(unsigned char)(pi2c->adr)|I2C_Read); // generate a start conditon
	
	if(IsAck(i2c)==I2C_NACK){					  // if no ACK, then generate a stop conditon
		I2C_Stop(i2c);
		return 0;
	}		
	
    *pi2c->pBuf++ = pi2c->len = I2C_RXOneByte(i2c);   // The first byte is data packet length 
	pi2c->len--;
		
	for(cnt=0; cnt<pi2c->len; cnt++){
		if(cnt == pi2c->len-1){
			I2C_AckEnable(i2c,0);                 // ACK disable 
			I2C_BufCount(i2c,BC_8bit);            // set number of data bits to 8
			*pi2c->pBuf++ = I2C_RXOneByte(i2c);
			I2C_BufCount(i2c,BC_1bit);            // tranmit 1-bit ACK
			I2C_TXOneByte(i2c,0xFF);              // write NACK data to 0xFF			
			I2C_AckEnable(i2c,1);                 // ACK enable 
			I2C_BufCount(i2c,BC_8bit);            // 8-bit received data 
		}
		else{
    		*pi2c->pBuf++ = I2C_RXOneByte(i2c); // 8-bit received data + 1-bit ACK
		}		
	}
	
	I2C_Stop(i2c);								 // generate a stop conditon
	
	return 1;

}

/* ------------------- end of file  ---------------------- */
