//*************************************************************************************
// file			: SQ7515_RTC.c
// version		: V1.0
// brief		: RTC related functions
// note			: RTC related functions are gathered in this subroutine (for SQ7515_OPMode\SQ7515_Sleep_Deep_Sleep_Mode_RTCINT_wake_up)
//**************************************************************************************
#include "main.h"

extern RTC_Perpetual_Calendar_Time rtc;                             
extern RTC_ALARM rtc_alarm;

unsigned char RTC_1sec_actFlag;			  // flag for second interrupt
unsigned char RTC_alarm_actFlag;		  // flag for alarm interrupt
unsigned char RTC_timeover_actFlag;	      // flag for overtime interrupt

//*********************************************************************
// name			: RTC_Setup()
// brief		: set current time
// note			:
//*********************************************************************
void RTC_Setup()
{
	PCKEN6_RTC = 1;	                       // enable RTC clock

	// set current time : 2019/11/18 Monday 15:06:50
	rtc.year = 19;
	rtc.mon = 11;
	rtc.week = 1;
	rtc.day = 18;
	rtc.hour = 15;
	rtc.min = 6;
	rtc.sec = 50;
	RTC_SetDate(&rtc);                  	// set RTC time
}
	
//*********************************************************************
// name			: Alarm_Setup()
// brief		: setup alarm time
// note			:
//*********************************************************************	
void Alarm_Setup()
{
	// set alarm time : 18 Monday 15:08:00
	rtc_alarm.week = 1|0x80;
	rtc_alarm.day = 18|0x80;
	rtc_alarm.hour = 15|0x80;
	rtc_alarm.min = 8;
	RTC_SetAlarm(&rtc_alarm);           	 // set alarm time 				
}

//*********************************************************************
// name			: RTC_Output()
// brief		: RTC clock output
// note			:
//*********************************************************************
void RTC_Output(void)
{
	unsigned char tmp;
	P2OE_P2  = 1;		                	// set P2.2 as output pin
	P2FC1_P2 = 0;			            	// set P2.2 to RTC0
	P2FC2_P2 = 1;
	
	RTCCR0_CLKOSEL = 0x00;                  // select the source clock of RTC clock output(CLKOUT) = RTC CLK = LXTAL/8 = 32.768 KHz/8 = 4.01 KHz
	RTCCR0_CLKOEN = 1;		                // enable RTC clock output  
}


//*********************************************************************
// name			: RTC_Init()
// brief		: RTC initialization
// note			:
//*********************************************************************
void RTC_Init()
{
	PCKEN6_RTC = 1;	                        // enable RTC clock

}


//*********************************************************************
// name			: RTC_IntEnable()
// brief		: enable RTC interrupt
// note			:
//*********************************************************************
void RTC_IntEnable(unsigned char intsel,unsigned char enable){
	__ASM("DI");							// disable all maskable interrupts
	
	IER_RTC = enable;		                
	RTCCR1 = intsel;
	
	__ASM("EI");							// enable all maskable interrupts
}

//*********************************************************************
// name			: RTC_IntDisable()
// brief		: disable RTC interrupt
// note			:
//*********************************************************************
void RTC_IntDisable(){
	__ASM("DI");							// disable all maskable interrupts
	IER_RTC = 0;		
	RTCCR1 =  0x00;              
	__ASM("EI");							// enable all maskable interrupts
}

//*********************************************************************
// name			: RTC_SetDate()
// brief		: set RTC time registers
// note			: it will transfer to BCD format
//*********************************************************************
void RTC_SetDate(RTC_Perpetual_Calendar_Time* prtc){
    RTCSEC = ((prtc->sec/10)<<4)|(prtc->sec%10);		    // RTC second,  0~59
    RTCMIN = ((prtc->min/10)<<4)|(prtc->min%10); 	        // RTC minute,  0~59
    RTCHR = ((prtc->hour/10)<<4)|(prtc->hour%10); 		    // RTC hour, 0~23
    RTCDAY = ((prtc->day/10)<<4)|(prtc->day%10); 	        // RTC day, 1~31
    RTCWDAY = (prtc->week&0x7); 	                        // RTC days of the week
    RTCMONTH = ((prtc->mon/10)<<4)|(prtc->mon%10); 	        // RTC month, 1~12
    RTCYEAR = ((prtc->year%100/10)<<4)|(prtc->year%10); 	// RTC year, 1~99
}


//*********************************************************************
// name			: RTC_Refresh()
// brief		: refresh RTC time
// note			: it will transfer to BCD format
//*********************************************************************
void RTC_Refresh(RTC_Perpetual_Calendar_Time* prtc){
	prtc->sec = (((RTCSEC&0x70)>>4)*10)+(RTCSEC&0x0F);
	prtc->min = (((RTCMIN&0x70)>>4)*10)+(RTCMIN&0x0F);
	prtc->hour = (((RTCHR&0x30)>>4)*10)+(RTCHR&0x0F);
	prtc->day = (((RTCDAY&0x30)>>4)*10)+(RTCDAY&0x0F);
	prtc->week = (RTCWDAY&0x0F);
	prtc->mon = (((RTCMONTH&0x10)>>4)*10)+(RTCMONTH&0x0F);
	prtc->year = (((RTCYEAR&0xF0)>>4)*10)+(RTCYEAR&0x0F);
}


//*********************************************************************
// name			: RTC_SetAlarm()
// brief		: set RTC alarm time
// note			: in BCD format
//*********************************************************************
void DRV_RTC_SetAlarm(RTC_ALARM* palarm) {
    // remember enable alarm interrupt first
    RTCALMIN = ((palarm->min/10)<<4)|(palarm->min%10); 	    // set RTC alarm minute
    RTCALHR = ((palarm->hour/10)<<4)|(palarm->hour%10);	    // set RTC alarm hour
    RTCALDAY = ((palarm->day/10)<<4)|(palarm->day%10); 	    // set RTC alarm day
    RTCALWDAY = ((palarm->week/10)<<4)|(palarm->week%10); 	// set RTC alarm day of the week
}


//*********************************************************************
// name			: RTC_Start()
// brief		: start RTC
// note			:
//*********************************************************************
void RTC_Start(void)
{
	RTCCR0_RTCEN = 1;		                  // start RTC
}


//*********************************************************************
// name			: RTC_Stop()
// brief		: stop RTC
// note			:
//*********************************************************************
void RTC_Stop(void)
{
	RTCCR0_RTCEN = 0;			              // stop RTC
}

//*********************************************************************
// name			: RTC_TimerSetAndEnable()
// brief		: set RTC function
// note			:
//*********************************************************************
void RTC_TimerSetAndEnable(void)
{
	RTCCR0 = 0x00;							  // RTC setting
	RTCCR1 |= 0x08;							  // enable RTC timeout interrupt (RTCCR1_TOIE = 1)

	//RTCTMRCR_CONT = 0;					  // set RTC to single mode
	RTCTMRCR_CONT = 1;						  // set RTC to continuous mode

	RTCTMRTO = 0x02;						  // RTC timeout value (must larger than 0)
}


//*********************************************************************
// name			: RTC_TimerStart()
// brief		: enable RTC timer function
// note			:
//*********************************************************************
void RTC_TimerStart(void)
{
	RTCTMRCR_TMREN = 1;						  // enable RTC timer function
}

//*********************************************************************
// name			: RTC_TimerStop()
// brief		: disable RTC timer function
// note			:
//*********************************************************************
void RTC_TimerStop(void)
{
	RTCTMRCR_TMREN = 0;						  // disable RTC timer function
	while(RTCTMRCR_TMRST != 0); 			  // make sure timer stop
}


//*********************************************************************
// name			: __interrupt IntRTC()
// brief		: RTC interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntRTC(void)
{
	__asm("NOP");					    	  // no operation
	
	//--- second interrupt ---
	if (RTCCR1_SECF == 1 )			    	  // second interrupt is generated
	{
		RTCCR1 = (RTCCR1&0x0F)|0x40;    	  // clear second interrupt flag
		
		RTC_1sec_actFlag = 1;				  // set RTC_1sec_actFlag to 1

	}
	
	
	//--- alarm interrupt ---
	if(RTCCR1_ALF == 1)						  // alarm interrupt is generated
	{
		RTCCR1 = (RTCCR1&0x0F)|0x20;		  // clear alarm interrupt flag
		
		
		RTC_alarm_actFlag = 1;				  // set RTC_alarm_actFlag to 1
	}
	
	
	//--- timer overtime interrupt ---
	if (RTCCR1_TOF == 1)			    	  // timer overtime interrupt is generated
	{
		RTCCR1 = (RTCCR1&0x0F)|0x80;		  // clear timer overtime interrupt flag
		
		RTC_timeover_actFlag = 1;				  // set RTC_timeover_actFlag to 1
	}

	
}


