//*************************************************************************************
// file			: SQ7515_ADC.c
// version		: V1.0 2022/05/18
// brief		: ADC related functions
// note			: ADC related functions are gathered in this subroutine
//**************************************************************************************

#include "main.h"


//*********************************************************************
// name			: ADC_Init()
// brief		: ADC initialization
// note			: 
//*********************************************************************
void ADC_Init(){
	
//====== ADC pin configuration initialization (select the analog signal input channel for AD conversion) ======
// SQ7515 has 12 pins available from AIN0 to AIN11

/*
    //--- AIN0 ---//	
    P1PU_P7 = 0;
    P1FC1_P7 = 1;                       // P1.7 is set to AIN0
    P1FC2_P7 = 1; 
*/

    //--- AIN1 ---//	[this pin is selected for this example]
    P1PU_P6 = 0;
    P1FC1_P6 = 1;                       // P1.6 is set to AIN1
    P1FC2_P6 = 1; 

/*
    //--- AIN2 ---//
    P1PU_P5 = 0;
    P1FC1_P5 = 1;                       // P1.5 is set to AIN2
    P1FC2_P5 = 1; 
*/

	
/*
    //--- AIN3 ---//
    P1PU_P4 = 0;
    P1FC1_P4 = 1;                       // P1.4 is set to AIN3
    P1FC2_P4 = 1; 
*/

/*
    //--- AIN4 ---//
    P1PU_P3 = 0;
    P1FC1_P3 = 1;                       // P1.3 is set to AIN4
    P1FC2_P3 = 1; 
*/

/*
    //--- AIN5 ---//
    P1PU_P2 = 0;
    P1FC1_P2 = 1;                       // P1.2 is set to AIN5
    P1FC2_P2 = 1; 
*/	
	
/*
    //--- AIN6 ---//
    P1PU_P1 = 0;
    P1FC1_P1 = 1;                       // P1.1 is set to AIN6
    P1FC2_P1 = 1; 
*/		

/*
    //--- AIN7 ---//
    P1PU_P0 = 0;
    P1FC1_P0 = 1;                       // P1.0 is set to AIN7
    P1FC2_P0 = 1; 
*/	

/*
    //--- AIN8 ---//	*Note:  another function of P5.3 pin is ADC reference voltage input (ADC_VREF) = VDD
    P5PU_P3 = 0;
    P5FC1_P3 = 1;                       // P5.3 is set to AIN8
    P5FC2_P3 = 1; 
*/	

/*
    //--- AIN9 ---//
    P5PU_P2 = 0;
    P5FC1_P2 = 1;                       // P5.2 is set to AIN9
    P5FC2_P2 = 1; 
*/	

/*
    //--- AIN10 ---//
    P5PU_P1 = 0;
    P5FC1_P1 = 1;                       // P5.1 is set to AIN10
    P5FC2_P1 = 1; 
*/	


/*
    //--- AIN11 ---//
    P5PU_P0 = 0;
    P5FC1_P0 = 1;                       // P5.0 is set to AIN11
    P5FC2_P0 = 1; 
*/	

//====== Function Enable ======        // this must be enabled first, the relevant staging settings later to be useful
    PCKEN6_ADC = 1;                     // enable ADC peripheral clock

//====== parameter setting ======
// Set ADC Convert Speed
    ADCCKDIV_ADCKDIV = 2;               // fsysclk/(2^2)

// after ADEN is set, the ADC needs to wait for clock warm up to allow the user to set ADCSR_ADRS to start the AD conversion
    ADCCR0_ADEN  = 1;                   // enable ADC
    ADCCR0_IRFEN = 0x01;                // set VDDA as Vref
    //ADCCR0_INTLV  = 1;                  // interrupt interval is set to wait until all AIN scans are completed before an interrupt is generated

    ADCCR0_AUTOPD = 0;                  // disable automatic power down after each ADC conversion
    
// set ADC operating mode
    ADCCR0_AMD   = 0x01;                // single mode

// ADC data buffer automatic overwrite
    //ADCSR_BUFOVR = 1;                 // ADC data buffer automatic overwrite

// select ADC input trigger event source	
    ADCCR1_EVSEL = 0x00;                // set event to ADC convertion

// Multiple AIN channel scanning
    ADCSCAN0 = 0x02;                    // set ADC scan channel to AIN1
    //ADCSCAN1 = 0x00; 

}


//*********************************************************************
// name			: ADC_IntEnable()
// brief		: enable ADC interrupt
// note			: 
//*********************************************************************
void ADC_IntEnable(){
    __ASM("DI");

    IER_ADC = 1;                        // enable ADC interrupt
    ADCCR0_INTEN = 1;                   // enable AD conversion finished interrupt

    __ASM("EI");

}

//*********************************************************************
// name			: ADC_IntDisable()
// brief		: disable ADC interrupt
// note			: 
//*********************************************************************
void ADC_IntDisable(){
    __ASM("DI");

    IER_ADC = 0;                        // disable ADC interrupt
    ADCCR0_INTEN = 0;                   // disable AD conversion finished interrupt

	__ASM("EI");

}

//*********************************************************************
// name			: ADC_Stop_and_Disable()
// brief		: disable ADC operation and forcibly stop ADC operation
// note			: 
//*********************************************************************
void ADC_Stop_and_Disable(){
    ADCCR0_AMD  = 0;                    // disable ADC operation and forcibly stop ADC operation
    ADCCR0_ADEN = 0;                    // disable ADC
}



//*********************************************************************
// name			: ADC_Convert_Single()
// brief		: perform ADC single mode conversion
// note			: 
//*********************************************************************
void ADC_Convert_Single(){
    //ADCCR0_AMD = 0x01;                  // set ADC operating mode: single mode
    while(!ADCSR_ADRDY);                // wait for ADC to be ready
    ADCSR_ADRS = 1;                     // start ADC conversion	
}

/*
//*********************************************************************
// name			: ADC_Convert_Repeat()
// brief		: Perform ADC repeat mode conversion and store the conversion value in ADCDRL and ADCDRH and average them
// note			: SAIN: select ADC input channel
//					AIN4		0x4
//					AIN5		0x5
//					AIN6		0x6
//					AIN7		0x7
//					AIN8		0x8
//					AIN9		0x9
//					AIN10		0xA
//
// *this example of the function is demonstrated repeated sampling within 18 times, and maximum and minimum values in the sampling process are not included in the average consideration
//*********************************************************************
unsigned int ADC_Convert_Repeat(unsigned char SAIN){
    uchar  i;
    ulong  adc_data = 0;
    uint   ad_min   = 0, ad_max = 4096;
    BIT_LIST_16 ADC_READ;

//====== parameter setting ======
	ADCCHSEL_CHSEL = SAIN;					// select ADC input channel
	ADCCR0_AMD = 0x03;                  	// set ADC operating mode:repeat mode
	
	while(!ADCSR_ADRDY);                	// wait for ADC to be ready
    ADCSR_ADRS = 1;                    		// start AD conversion

    for( i=0; i<(ad_test_times+2); i++ )	// repeat sampling 18 times
    {
        while( !ADCSR_EOCF ){ CLR_WDT; } 	// wait for AD conversion complete
		ADCSR_EOCF = 1;             		// clear ADC conversion completion flag		
        ADC_READ.byte[0] = ADCDRL;      	// when reading ADC value,  read low byte first and then read high byte
        ADC_READ.byte[1] = ADCDRH;      	// when reading ADC value,  read low byte first and then read high byte
        adc_data += ADC_READ.word;
        if( ADC_READ.word > ad_min ) ad_min = ADC_READ.word;
        if( ADC_READ.word < ad_max ) ad_max = ADC_READ.word;
    }
    ADCSR_ADRS = 0;                    		// stop AD conversion

    adc_data -= ad_max;                 	// subtract the maximum value during sampling
    adc_data -= ad_min;                 	// subtract the minimum value during sampling
    
	adc_data /= ad_test_times;				// average

	return (unsigned int)adc_data;
}
*/


//*********************************************************************
// name			: ADC_ConvertDone()
// brief		: ADC conversion completed
// note			: 
//*********************************************************************
char ADC_ConvertDone(){
    if(ADCSR_EOCF){
        return 1;                       // if ADC conversion completed, return 1
    }
    else{
        return 0;                       // if ADC conversion not completed, return 0
    }
}

//*********************************************************************
// name			: ADC_Read_Value()
// brief		: readout ADC value    [for single mode]
// note			: chsel : AIN setting value  
// 						AIN4	       0x4
// 						AIN5	       0x5
// 						AIN6	       0x6
// 						AIN7	       0x7
// 						AIN8	       0x8
// 						AIN9	       0x9
// 						AIN10	   	   0xA
//*********************************************************************
long ADC_Read_Value(unsigned char chsel){
    unsigned int adc_value = 0;

    ADCCHSEL = chsel;                   // select ADC input channel
	
    adc_value |= ADCDRL;                // when reading ADC value,  read low byte first and then read high byte

    adc_value |= (unsigned int)ADCDRH<<8;

    return (long)adc_value;
	
}


//*********************************************************************
// name			: __interrupt IntADC()
// brief		: ADC interrupt subroutine
// note			:
//*********************************************************************
void __interrupt IntADC(void){
    __asm("NOP");                       // no operation
}

/* ------------------------- end of file ---------------------------- */
