//*********************************************************************
// file			: MQ6835_I2C.c
// version		: V1.0 2019/04/19
// brief		: I2C related functions
// note			: I2C related functions are gathered in this subroutine
//*********************************************************************

#include "main.h"

User_I2C  Flow;
extern uchar wBuf[16];

//*********************************************************************
// name			: I2C_SetSlaveAddr
// brief		: set slave address
// note			:
//*********************************************************************
void I2C_SetSlaveAddr(uchar addr)
{
	I2C0AR = addr & 0xfe;				// SA0[7:1]: Slave address setting
										// 0Xfe	1111 1110		
										// &0xfe   : and clear bit0  
}

//*********************************************************************
// name			: I2C_Init
// brief		: enable I2C and start I2C
// note			: initial
//*********************************************************************
void I2C_Init(void)
{	//====== pin initialization ======
	P2CR_P3  = 0;						// set P23 as input pin
	P2CR_P4  = 0;						// set P24 as input pin
	//====== enable function ======
	POFFCR1_SBI0EN  = 1;				// this register must be enable first and then the related registers settings will work
	
	while( P2PRD_P3==0 || P2PRD_P4==0 ){// wait for high level
		CLR_WDT; 						// clear the watchdog timer
	}	
	//====== parameter setting ======
	P2OUTCR_P3 = 1;						//set P23 to open drain output
	P2OUTCR_P4 = 1;						//set P24 to open drain output
	
	P2DR_P3    = 0; 				    // set P23 to output low
	P2DR_P4    = 0;						// set P24 to output low
	
	P2FC_P3    = 1;						// set P23 to special function pin
	P2FC_P4    = 1;						// set P24 to special function pin
	
	P2CR_P3    = 1; 					// set P23 as output pin
	P2CR_P4    = 1; 					// set P24 as output pin

//-----------------------------------------------------------------------
//------------bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ----
//--- SBI0CR2 <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>
//            <MST>   1 : Master
//                    0 : Slave
// 			  <TRX>   1 : Transmitter / 0 : Receiver
// 			  <BB>    1 : generate start condition
//       	          0 : generate stop condition
//			  <PIN>	  1 : cancel interrupt service request
//
//			  <SBIM>  1 : Serial bus interface mode / 0:Port mode
//            <SWRST> software reset start bit  	

	SBI0CR2 = 0x18;						// 0001 1000
	SBI0CR1 = 0x10;						// 0001 0000

}

//*********************************************************************
// name			: I2C_IntEnable
// brief		: enable I2C interrupt
// note			: after execution, Interrupt Master Allowed Flag (IMF) will be enabled
//*********************************************************************
void I2C_IntEnable(void)
{
	DI;
	IL_INTSBI = 0;
	EI_INTSBI = 1;
	EI;
}


//*********************************************************************
// name			: I2C_IntDisable 
// brief		: disable I2C interrupt
// note			: after execution, Interrupt Master Allowed Flag (IMF) will be enabled
//*********************************************************************
void I2C_IntDisable(void)
{
	DI;
	EI_INTSBI = 0;
	IL_INTSBI = 0;	
	EI;
}


//*********************************************************************
// name			: I2C_Start
// brief		: generate start condition
// note			:
//*********************************************************************
//------------bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ----
//--- SBI0CR2 <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>
//            <MST>   1 : Master
//                    0 : Slave
// 			  <TRX>   1 : Transmitter / 0 : Receiver
// 			  <BB>    1 : generate start condition
//       	          0 : generate stop condition
//			  <PIN>	  1 : cancel interrupt service request
//
//			  <SBIM>  1 : Serial bus interface mode / 0:Port mode
//            <SWRST> software reset start bit 

void I2C_Start(uchar data, uchar address)
{
	uchar tWait = 0;

	while( SBI0SR2_BB == 1 ){}			// SBI0SR2.BB: Bus status confirms is free ; 1 :busy  0: free

	SBI0CR1 = 0x10;						// ***SBI0CR1 needs to be set for each I2C_Start.
//****** Write data to the registers before the start condition is generated ******
	SBI0DBR = data;						// set slave Device ID
	SBI0CR2 = 0xf8;						// Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
										// 1111 1000
										// Set master,transmitter,generate start condition,cancel interrupt service request and serial bus interface mode
	
	
	while( SBI0SR2_PIN == 1 ){}			// wait for releasing interrupt service request	
	while( SBI0SR2_LRB == 1 ){			// check <LRB> check ACK signal from slave device									
		tWait++;						// if no response, SCK is always low level
		if( tWait>250 ){				// jump out of loop if no response for too long		
			I2C_Stop();					// I2C stop -> release SCK
			break;
		}
	}

//... set slave address..................
	tWait = 0;
	SBI0DBR = address;					// set slave address
	while( SBI0SR2_PIN == 1 ){}			// wait for releasing interrupt service request
	while( SBI0SR2_LRB == 1 ){			// wait for last received bit is "0"										
		tWait++;						// if no response, SCK is always low level
		if( tWait>250 ){				// jump out of loop if no response for too long		
			I2C_Stop();					// I2C stop -> release SCK
			break;
		}
	}
}


//*********************************************************************
// name			: I2C_Restart 
// brief		: generate restart condition
// note			:
//*********************************************************************
//------------bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ----
//--- SBI0CR2 <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>
//            <MST>   1 : Master
//                    0 : Slave
// 			  <TRX>   1 : Transmitter / 0 : Receiver
// 			  <BB>    1 : generate start condition
//       	          0 : generate stop condition
//			  <PIN>	  1 : cancel interrupt service request
//
//			  <SBIM>  1 : Serial bus interface mode / 0:Port mode
//            <SWRST> software reset start bit 

void I2C_Restart(uchar data)
{
	uchar tWait = 0;

	SBI0CR2 = 0x18;						// 0001 1000
										// Set master,transmitter,generate start condition,cancel interrupt service request and serial bus interface mode
										
	while( SBI0SR2_BB == 1 ){}			// SBI0SR2.BB: Bus status confirms is free ; 1 :busy  0: free
	while( SBI0SR2_LRB == 0 ){}			// wait <LRB>=1**********************

//****** Write data to the registers before the start condition is generated ******
	SBI0DBR = data;
	SBI0CR2 = 0xf8;						// Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
	while( SBI0SR2_PIN == 1 ){}			// wait for releasing interrupt service request
	while( SBI0SR2_LRB == 1 ){			// wait for last received bit is "0"										
		tWait++;						// if no response, SCK is always low level
		if( tWait>250 ){				// jump out of loop if no response for too long		
			I2C_Stop();					// I2C stop -> release SCK
			break;
		}
	}
}


//*********************************************************************
// name			: I2C_Stop 
// brief		: generate stop condition
// note			:
//*********************************************************************
//------------bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ----
//--- SBI0CR2 <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>
//            <MST>   1 : Master
//                    0 : Slave
// 			  <TRX>   1 : Transmitter / 0 : Receiver
// 			  <BB>    1 : generate start condition
//       	          0 : generate stop condition
//			  <PIN>	  1 : cancel interrupt service request
//
//			  <SBIM>  1 : Serial bus interface mode / 0:Port mode
//            <SWRST> software reset start bit 

void I2C_Stop(void)
{
	SBI0CR2 = 0xd8;						// Set SBI0CR2<MST>, <TRX> and <PIN> to "1" and SBI0CR2<BB> to "0"
										// 1101 1000
										// Set master,transmitter,generate stop condition,cancel interrupt service request and Portģʽ
										
	while( SBI0SR2_BB == 1 ){}			// SBI0SR2.BB: Bus status confirms is free ; 1 :busy  0: free
}


//*********************************************************************
// name			: I2C_ByteWrite 
// brief		: byte write * len
// note			:
//*********************************************************************
void I2C_ByteWrite(uchar *data, uchar len)
{
	uchar i, tWait;

	for( i=0; i<len; i++ ){	
		tWait = 0;
		SBI0DBR = *data;
		data++;
		while( SBI0SR2_PIN == 1 ){}		// wait for releasing interrupt service request
		while( SBI0SR2_LRB == 1 ){		// wait for last received bit is "0"
			tWait++;	
			if( tWait>250 ){			// jump out of loop if no response for too long			
				I2C_Stop();				// I2C stop -> release SCK
				break;
			}
		}
	}

}


//*********************************************************************
// name			: I2C_ByteRead 
// brief		: byte read * len
// note			:
//*********************************************************************
void I2C_ByteRead(uchar *data, uchar len)
{
	uchar i;

	len -= 1;

	for( i=0; i<len; i++ ){
		SBI0DBR   = 0x00;				// write dummy data to set PIN = 1
		while( SBI0SR2_PIN == 1 ){}		// wait for releasing interrupt service request
		*data = SBI0DBR;
		data++;
	}

	SBI0CR1_ACK = 0;					// ACK=0
	SBI0DBR   = 0x00;					// write dummy data to set PIN = 1
	while( SBI0SR2_PIN == 1 ){}			// wait for releasing interrupt service request
	*data = SBI0DBR;

//	Termination of data transfer by master mode
//	Sent 1 bit negative acknowledge............
	SBI0CR1_BC = 1;						// BC, bit count = 1
	SBI0DBR   = 0x00;					// write dummy data to set PIN = 1
	while( SBI0SR2_PIN == 1 ){}			// wait for releasing interrupt service request
}




//*********************************************************************
// file			: IntSBI
// brief		: I2C interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntSBI(void)
{
   BIT_SBI0SR2 I2C_Flag;
    
    *((uchar *)(&I2C_Flag)) = SBI0SR2;
    
//----------------------------------------------------------//
//****** Receiver at slave mode ****************************//
//----------------------------------------------------------//
    if( I2C_Flag.TRX == 0 ){                            // TRX: Transmitter/Receiver selection status monitor.==> 0).Receiver, 1).Transmitter   
        if( I2C_Flag.AAS == 1 ){                        // AAS: Slave address match detection monitor       
            Flow.sFlag.match_addr = 1;                  // first address match detection
            Flow.iByte            = 0;
            Flow.sFlag.R1W0       = I2C_Flag.LRB;       // 1:read, 0:write.
            SBI0DBR               = 0;                  // Write the dummy data (0x00) to the SBI0DBR to set SBI0CR2<PIN> to "1"

        }
        else{       
            if( Flow.sFlag.match_addr == 1 ){           // get address first            
                Flow.iAddr_st             = SBI0DBR;
                Flow.iAddr_pt             = Flow.iAddr_st;
                Flow.iByte                = Flow.iAddr_st;
                Flow.sFlag.match_addr = 0;
                SBI0DBR              = 0;               // Write the dummy data (0x00) to the SBI0DBR to set SBI0CR2<PIN> to "1"
            }
            else{           
                wBuf[Flow.iByte] = SBI0DBR;             // get write data from master
                Flow.iByte++;
                Flow.iByte &= 0x0f;                     // max. 16 byte: then overwrite data at 1st has get
                SBI0DBR    = 0;                         // Write the dummy data (0x00) to the SBI0DBR to set SBI0CR2<PIN> to "1"
            }
        }
    }

//----------------------------------------------------------//
//****** Transmitter at slave mode *************************//
//----------------------------------------------------------//
    else
    {
        if( I2C_Flag.AAS == 1 ){                         // AAS: Slave address match detection monitor       
            SBI0DBR = wBuf[Flow.iByte];
            Flow.iByte++;
        }
        else{
            if( I2C_Flag.LRB == 1 ){                     // LRB: Last received bit monitor           
        	//----------------------------------------------------------//
        	//****** Master non-ACK for last byte **********************//
        	//----------------------------------------------------------//
				//SBI0DBR = 0;                             // Write the dummy data (0x00) to the SBI0DBR to set SBI0CR2<PIN> to "1"
				SBI0CR2 = 0x58;                          // PIN = 1
			    SBI0CR2 = 0x18;                          // TRX = 0
            }
            else{
				SBI0DBR = wBuf[Flow.iByte];
				Flow.iByte++;			
            }
        }
    } 
    
}






