//*********************************************************************
// file			: MQ6832_I2C.c
// version		: V1.0
// brief		: I2C related functions
// note			: I2C related functions are gathered in this subroutine
//*********************************************************************

#include "main.h"


//*********************************************************************
// name			: I2C_SetSlaveAddr
// brief		: set slave address
// note			:
//*********************************************************************
void I2C_SetSlaveAddr(uchar addr)
{
	I2C0AR = addr & 0xfe;				// SA0[7:1]: Slave address setting
										// &0xfe   : and clear bit0
}


//*********************************************************************
// name			: I2C_Init
// brief		: enable I2C and start I2C
// note			:
//*********************************************************************
void I2C_Init(void)
{
	P2CR_P3  = 0;
	P2CR_P4  = 0;
	POFFCR1_SBI0EN  = 1;				// SBI0 Enable
	while( P2PRD_P3==0 || P2PRD_P4==0 ){ CLR_WDT; }	// wait for high level
	P2DR_P3    = 0;
	P2DR_P4    = 0;
	P2FC_P3    = 1;
	P2FC_P4    = 1;
	P2OUTCR_P3 = 1;
	P2OUTCR_P4 = 1;
	P2CR_P3    = 1;
	P2CR_P4    = 1;

//----------------------------------//
	SBI0CR2 = 0x18;
	SBI0CR1 = 0x10;

}

//*********************************************************************
// name			: I2C_IntEnable
// brief		: enable I2C interrupt
// note			: after execution, Interrupt Master Allowed Flag (IMF) will be enabled
//*********************************************************************
void I2C_IntEnable(void)
{
	DI;
	IL_INTSBI = 0;
	EI_INTSBI = 1;
	EI;
}


//*********************************************************************
// name			: I2C_IntDisable 
// brief		: disable I2C interrupt
// note			: after execution, Interrupt Master Allowed Flag (IMF) will be enabled
//*********************************************************************
void I2C_IntDisable(void)
{
	DI;
	EI_INTSBI = 0;
	IL_INTSBI = 0;	
	EI;
}


//*********************************************************************
// name			: I2C_Start
// brief		: generate start condition
// note			:
//*********************************************************************
void I2C_Start(uchar data, uchar address)
{
	uchar tWait = 0;

	while( SBI0SR2_BB == 1 ){}			// SBI0SR2.BB: confirms bus is free

	SBI0CR1 = 0x10;						// SBI0CR1 needs to be set for each I2C_Start.
//****** Write data to the registers before the start condition is generated ******
	SBI0DBR = data;						// set slave Device ID
	SBI0CR2 = 0xf8;						// Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
	while( SBI0SR2_PIN == 1 ){}			// wait for releasing interrupt service request	
	while( SBI0SR2_LRB == 1 ){			// wait for last received bit is "0"									
		tWait++;						// if no response, SCK is always low level
		if( tWait>250 ){				// jump out of loop if no response for too long		
			I2C_Stop();					// I2C stop -> release SCK
			break;
		}
	}

//... set slave address..................
	tWait = 0;
	SBI0DBR = address;					// set slave address
	while( SBI0SR2_PIN == 1 ){}			// wait for releasing interrupt service request
	while( SBI0SR2_LRB == 1 ){			// wait for last received bit is "0"										
		tWait++;						// if no response, SCK is always low level
		if( tWait>250 ){				// jump out of loop if no response for too long		
			I2C_Stop();					// I2C stop -> release SCK
			break;
		}
	}
}


//*********************************************************************
// name			: I2C_Restart 
// brief		: generate restart condition
// note			:
//*********************************************************************
void I2C_Restart(uchar data)
{
	uchar tWait = 0;

	SBI0CR2 = 0x18;
	while( SBI0SR2_BB == 1 ){}			// SBI0SR2.BB: confirms bus is free
	while( SBI0SR2_LRB == 0 ){}			// wait <LRB>=1**********************

//****** Write data to the registers before the start condition is generated ******
	SBI0DBR = data;
	SBI0CR2 = 0xf8;						// Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
	while( SBI0SR2_PIN == 1 ){}			// wait for releasing interrupt service request
	while( SBI0SR2_LRB == 1 ){			// wait for last received bit is "0"										
		tWait++;						// if no response, SCK is always low level
		if( tWait>250 ){				// jump out of loop if no response for too long		
			I2C_Stop();					// I2C stop -> release SCK
			break;
		}
	}
}


//*********************************************************************
// name			: I2C_Stop 
// brief		: generate stop condition
// note			:
//*********************************************************************
void I2C_Stop(void)
{
	SBI0CR2 = 0xd8;						// Set SBI0CR2<MST>, <TRX> and <PIN> to "1" and SBI0CR2<BB> to "0"
	while( SBI0SR2_BB == 1 ){}			// SBI0SR2.BB: confirms bus is free
}


//*********************************************************************
// name			: I2C_ByteWrite 
// brief		: byte write * len
// note			:
//*********************************************************************
void I2C_ByteWrite(uchar *data, uchar len)
{
	uchar i, tWait;

	for( i=0; i<len; i++ ){	
		tWait = 0;
		SBI0DBR = *data;
		data++;
		while( SBI0SR2_PIN == 1 ){}		// wait for releasing interrupt service request
		while( SBI0SR2_LRB == 1 ){		// wait for last received bit is "0"
			tWait++;	
			if( tWait>250 ){			// jump out of loop if no response for too long			
				I2C_Stop();				// I2C stop -> release SCK
				break;
			}
		}
	}

}


//*********************************************************************
// name			: I2C_ByteRead 
// brief		: byte read * len
// note			:
//*********************************************************************
void I2C_ByteRead(uchar *data, uchar len)
{
	uchar i;

	len -= 1;

	for( i=0; i<len; i++ ){
		SBI0DBR   = 0x00;				// write dummy data to set PIN = 1
		while( SBI0SR2_PIN == 1 ){}		// wait for releasing interrupt service request
		*data = SBI0DBR;
		data++;
	}

	SBI0CR1_ACK = 0;					// ACK=0
	SBI0DBR   = 0x00;					// write dummy data to set PIN = 1
	while( SBI0SR2_PIN == 1 ){}			// wait for releasing interrupt service request
	*data = SBI0DBR;

//	Termination of data transfer by master mode
//	Sent 1 bit negative acknowledge............
	SBI0CR1_BC = 1;						// BC, bit count = 1
	SBI0DBR   = 0x00;					// write dummy data to set PIN = 1
	while( SBI0SR2_PIN == 1 ){}			// wait for releasing interrupt service request
}




//*********************************************************************
// file			: IntSBI
// brief		: I2C interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntSBI(void)
{
    
}






