//******************************************
// file			: MQ6825_LCD.c
// version		: V1.1 2021/02/25
// brief		: LCD related functions
// note			: LCD related functions are gathered in this subroutine
//******************************************

#include "main.h"

//---------------------------------------------------------------------------------------
// from 0 to 10 and from A to F, total 16 digits, alphabetic display
// corresponding to LCD GDC03077 8com*14seg display
//                               			.egf                                   			    dcba
// 0 corresponding to GDC03077 fge segment: 0101 0x05  ,corresponding to GDC03077 abcd segment: 1111 0x0F
// 1 corresponding to GDC03077 fge segment: 0000 0x00  ,corresponding to GDC03077 abcd segment: 0110 0x06
// 2 corresponding to GDC03077 fge segment: 0110 0x06	,corresponding to GDC03077 abcd segment: 1011 0x0B
// 3 corresponding to GDC03077 fge segment: 0010 0x02	,corresponding to GDC03077 abcd segment: 1111 0x0F
// 4 corresponding to GDC03077 fge segment: 0011 0x03	,corresponding to GDC03077 abcd segment: 0110 0x06
// 5 corresponding to GDC03077 fge segment: 0011 0x03	,corresponding to GDC03077 abcd segment: 1101 0x0D
// 6 corresponding to GDC03077 fge segment: 0111 0x07	,corresponding to GDC03077 abcd segment: 1101 0x0D
// 7 corresponding to GDC03077 fge segment: 0001 0x01	,corresponding to GDC03077 abcd segment: 0111 0x07

// 8 corresponding to GDC03077 fge segment: 0111 0x07  ,corresponding to GDC03077 abcd segment: 1111 0x0F
// 9 corresponding to GDC03077 fge segment: 0011 0x03  ,corresponding to GDC03077 abcd segment: 1111 0x0F
// A corresponding to GDC03077 fge segment: 0111 0x07  ,corresponding to GDC03077 abcd segment: 0111 0x07
// B corresponding to GDC03077 fge segment: 0111 0x07  ,corresponding to GDC03077 abcd segment: 1100 0x0C
// C corresponding to GDC03077 fge segment: 0101 0x05  ,corresponding to GDC03077 abcd segment: 1001 0x09
// D corresponding to GDC03077 fge segment: 0110 0x06  ,corresponding to GDC03077 abcd segment: 1110 0x0E
// E corresponding to GDC03077 fge segment: 0111 0x07  ,corresponding to GDC03077 abcd segment: 1001 0x09
// F corresponding to GDC03077 fge segment: 0111 0x07  ,corresponding to GDC03077 abcd segment: 0001 0x01
//--------------------------------------------------------------------------------------

const uchar DIG_CODE_fge[16]={  0x05, 0x00, 0x06, 0x02, 0x03, 0x03, 0x07, 0x01,
                                0x07, 0x03, 0x07, 0x07, 0x05, 0x06, 0x07, 0x07 };

const uchar DIG_CODE_abcd[16]={	0x0F, 0x06, 0x0B, 0x0F, 0x06, 0x0D, 0x0D, 0x07,
                                0x0F, 0x0F, 0x07, 0x0C, 0x09, 0x0E, 0x09, 0x01 };

//	GDC03077 8com*14seg


//*********************************************************************
// name			: LCD_Init
// brief		: LCD initialization
// note			:
//*********************************************************************
void LCD_Init(void)
{
//====== pin configuration ======
// the pins used for the LCD function must be set as input pins
// this example only use from COM0 to COM7 and from SEG0 to SEG13

    P9CR_P0 = 0;                        // COM  0
    P9CR_P1 = 0;                        // COM  1
    P7CR_P2 = 0;                        // COM  2
    P7CR_P3 = 0;                        // COM  3
    P8CR_P2 = 0;                        // COM  4 <== or ==> SEG  28
    P8CR_P3 = 0;                        // COM  5 <== or ==> SEG  29
    P2CR_P3 = 0;                        // COM  6 <== or ==> SEG  30
    P2CR_P4 = 0;                        // COM  7 <== or ==> SEG  31

    P2CR_P5 = 0;                        // SEG  0
    P2CR_P6 = 0;                        // SEG  1
    P2CR_P7 = 0;                        // SEG  2
    P7CR_P7 = 0;                        // SEG  3
    P7CR_P5 = 0;                        // SEG  4
    P7CR_P0 = 0;                        // SEG  5
    P7CR_P6 = 0;                        // SEG  6
    P7CR_P4 = 0;                        // SEG  7

    P4CR_P7 = 0;                        // SEG  8
    P4CR_P6 = 0;                        // SEG  9
    P4CR_P5 = 0;                        // SEG 10
    P4CR_P4 = 0;                        // SEG 11
    P2CR_P0 = 0;                        // SEG 12
    P2CR_P1 = 0;                        // SEG 13
	
	//---------------------------------------------
	/*															
    P2CR_P2 = 0;                        // SEG 14
    P8CR_P4 = 0;                        // SEG 15

    P8CR_P5 = 0;                        // SEG 16
    P8CR_P6 = 0;                        // SEG 17
    P8CR_P7 = 0;                        // SEG 18
    P9CR_P2 = 0;                        // SEG 19
    P9CR_P3 = 0;                        // SEG 20
    P5CR_P7 = 0;                        // SEG 21
    P5CR_P6 = 0;                        // SEG 22
    P5CR_P5 = 0;                        // SEG 23

    P5CR_P4 = 0;                        // SEG 24
    P5CR_P3 = 0;                        // SEG 25
    P5CR_P2 = 0;                        // SEG 26
    P5CR_P1 = 0;                        // SEG 27
	*/

//====== enable function ======
    POFFCR2_LCDEN = 1;

//====== parameter setting ======

    LCDCR1_DUTY   = 0x05;               // 1/8 duty 1/4 bias
//  LCDCR1_DUTY   = 0x06;               // 1/8 duty 1/3 bias
//  LCDCR1_DUTY   = 0x07;               // 1/8 duty 1/2 bias
    LCDCR1_SLF    = 0x03;               // Fcgck/2^15
    LCDCR2_LDRS   = 1;					// select low driving current source

    LCDBUFxx_init_seq();				// enable all LCD buffer
	

    LCDCR3 = 0xff;                      // Enable COM_EN[7:0]

    LCDCR4 = 0xff;                      // Enable SEGH_EN[ 7: 0]
    LCDCR5 = 0x3f;                      // Enable SEGH_EN[15: 8]
	/*
    LCDCR6 = 0xff;                      // Enable SEGH_EN[23:16]
    LCDCR7 = 0x0f;                      // Enable SEGH_EN[31:24]
	*/
}



//**************************************************************************************
// name			: LCD_Power_Control
// brief		: set LCD power source and bias
// note			: LCDVCR1_VCR4_0 parameters have a total 32 levels to choose. 0 for the deepest and 31 for the lightest
//			  LCD text depth according to different LCD will be different, the user can adjust the LCDVCR1_VCR4_0 parameters
//**************************************************************************************
//void LCD_Power_Control(uchar ki, uchar z1, uchar z2)
//{
//  //---------------- set LCD power source --------------------------------------
//    //***check LCD operating voltage and VDD
//    LCDCR2_VLCSEL = 0;					// select VDD as the LCD power source
//	
//	//LCDCR2_VLCSEL = 1;				// select P71 as the LCD power source
//	
// //---------LCD bias ratio unadjusted to make LCD darkest color -------------------------
//	//LCDVCR1 = 0;						// adjust bias to let LCD display become darkest color
//
//
//
//	
// //---------LCD bias adjustment to make LCD darker or lighter -------------------------
//	
// /* ---------------------- global variables ------------------------------ */
//	z1=z2=2;						    // initialize repeat times	
//
// //-----------------------------------------------------------------
// //the following are sample steps for LCD text shades [dark=>light,dark=>light,light=>dark,light=>dark]
// //-----------------------------------------------------------------
//    // 1.gradually change all LCD text from the darkest color to the lightest color and repeat twice
//	for(z1=2; z1>0; --z1)
//	{
//	
//		for(ki=0; ki<32; ki++)
//		{
//			LCDVCR1_VCR4_0 = ki;		// LCD bias adjustment
//			delay_times(t_1ms, 250);	// delay 250ms
//		
//		}
//		
//	}	
//
//
//    // 2.gradually change all LCD text from the lightest color to the darkest color and repeat twice
//	for(z2=2; z2>0; --z2)
//	{	
//		for(ki=31; ki>0; ki--)
//		{
//			LCDVCR1_VCR4_0 = ki;		// LCD bias adjustment
//			delay_times(t_1ms, 250);	// delay 250ms
//		}
//	}
//		
//			
//	delay_times(t_1ms, 100);			// delay 100ms
//	
// //---------------------------------------------------------------------
// 
// 
// //-----------------------------------------------------------------
// //the following is an example of LCD text depth steps [light=>dark, light=>dark, dark=>light, dark=>light]
// //-----------------------------------------------------------------
// 
///*
//	// 1.gradually change all LCD text from the lightest color to the darkest color and repeat twice
//	for(z1=2; z1>0; --z1)
//	{
//		for(ki=31; ki>0; ki--)
//		{
//			LCDVCR1_VCR4_0 = ki;		// LCD bias adjustment
//			delay_times(t_1ms, 250);	// delay 250ms
//		}
//		
//	}	
//
//
//	// 2.gradually change all LCD text from the darkest color to the lightest color and repeat twice
//	for(z2=2; z2>0; --z2)
//	{	
//		for(ki=0; ki<32; ki++)
//		{
//			LCDVCR1_VCR4_0 = ki;		// LCD bias adjustment
//			delay_times(t_1ms, 250);	// delay 250ms
//
//		}
//	}
//		
//			
//	delay_times(t_1ms, 100);			// delay 100ms
//	//-------------------------------------------------------------------
//*/
//	
//	
//	
//
//}
//*/
//
//
//
////*********************************************************************
//// name			: LCD_Power_Control_08
//// brief			: adjust bias to let LCD display become lighter color 
//// note			: LCDVCR1_VCR4_0 parameters have a total 32 levels to choose. 0 for the deepest and 31 for the lightest
////*********************************************************************
//void LCD_Power_Control_08()
//{
//
//	LCDVCR1_VCR4_0 = 0x08;				// LCD bias adjustment
//	
//}
//
////*********************************************************************
//// name			: LCD_Power_Control_10
//// brief			: adjust bias to let LCD display become lighter color
//// note			: LCDVCR1_VCR4_0 parameters have a total 32 levels to choose. 0 for the deepest and 31 for the lightest
////*********************************************************************
//void LCD_Power_Control_10()
//{
//
//	LCDVCR1_VCR4_0 = 0x10;				// LCD bias adjustment
//	
//}
//

//*********************************************************************
// name			: LCD_Start
// brief		: Start LCD
// note			: 
//*********************************************************************
void LCD_Start(void)
{
    LCDCR1_EDSP = 1;					// start LCD
}

//*********************************************************************
// name			: LCD_Stop
// brief		: Stop LCD
// note			: 
//*********************************************************************
void LCD_Stop(void)
{
    LCDCR1_EDSP = 0;					// disable LCD display
}



//*********************************************************************
// name			: LCDBUFxx_init_seq
// brief		: enable all LCD buffer
// note			: MQ6825 (8com*28seg) (4com*32seg)
//*********************************************************************
void LCDBUFxx_init_seq(void)
{
    LCDBUF00 = 0xFF;
    LCDBUF01 = 0xFF;
    LCDBUF02 = 0xFF;
    LCDBUF03 = 0xFF;
    LCDBUF04 = 0xFF;
    LCDBUF05 = 0xFF;
    LCDBUF06 = 0xFF;
    LCDBUF07 = 0xFF;
    LCDBUF08 = 0xFF;
    LCDBUF09 = 0xFF;
    LCDBUF10 = 0xFF;
    LCDBUF11 = 0xFF;
    LCDBUF12 = 0xFF;                    // for SEG28 & SEG12
    LCDBUF13 = 0xFF;                    // for SEG29 & SEG13
		
										// the following example only uses from SEG0 to SEG13
    //---------------------------------------------------------------										
	/*
    LCDBUF14 = 0xFF;                    // for SEG30 & SEG14
    LCDBUF15 = 0xFF;                    // for SEG31 & SEG15
    LCDBUF16 = 0xFF;
    LCDBUF17 = 0xFF;
    LCDBUF18 = 0xFF;
    LCDBUF19 = 0xFF;
    LCDBUF20 = 0xFF;
    LCDBUF21 = 0xFF;
    LCDBUF22 = 0xFF;
    LCDBUF23 = 0xFF;
    LCDBUF24 = 0xFF;
    LCDBUF25 = 0xFF;
    LCDBUF26 = 0xFF;
    LCDBUF27 = 0xFF;
	*/
}


//*********************************************************************
// name			: Dig_Decode
// brief		: digital display for GDC03077
// note			: 
//*********************************************************************
void Dig_Decode(uchar digN, uchar Num)			// digN means drawing number, but Num means the number of digits and letters to be displayed
{
    uchar NumFGE=0, NumABCD=0, NumMask;

    if( digN/5 )								// drawing number 5~10 (>=5)
    {
        NumMask  = 0xf0;						// use(NumMask=1111 0000)
        NumFGE   = DIG_CODE_fge[Num];
        NumABCD  = DIG_CODE_abcd[Num];
    }
    else										// drawing number 1~4 (<5)
    {
        NumMask  = 0x0f;						// use(NumMask=0000 1111)
        NumFGE   = DIG_CODE_fge[Num]  << 4;     // shift left 4 bits
        NumABCD  = DIG_CODE_abcd[Num] << 4;     // shift left 4 bits
		
    }

    switch(digN){
        case 1:									// drawing number 1(use SEG0 and SEG1)
            LCDBUF00 &= NumMask;				// NumMask = 0000 1111
            LCDBUF01 &= NumMask;				// NumMask = 0000 1111
            LCDBUF00 |= NumFGE;					// SEG0
            LCDBUF01 |= NumABCD;				// SEG1
            break;
        case 2:									// drawing number 2(use SEG2 and SEG3)
            LCDBUF02 &= (NumMask|0x80);			// NumMask = 1000 1111 (0000 1111 or 1000 0000) 
            LCDBUF03 &= NumMask;				// NumMask = 0000 1111
            LCDBUF02 |= NumFGE;					// SEG2
            LCDBUF03 |= NumABCD;				// SEG3
            break;
        case 3:									// drawing number 3(use SEG4 and SEG5)
            LCDBUF04 &= (NumMask|0x80);			// NumMask = 1000 1111 (0000 1111 or 1000 0000) 
            LCDBUF05 &= NumMask;				// NumMask = 0000 1111
            LCDBUF04 |= NumFGE;					// SEG4
            LCDBUF05 |= NumABCD;				// SEG5
            break;
        case 4:									// drawing number 4(use SEG6 and SEG7)
            LCDBUF06 &= (NumMask|0x80);			// NumMask = 1000 1111 (0000 1111 or 1000 0000) 
            LCDBUF07 &= NumMask;				// NumMask = 0000 1111
            LCDBUF06 |= NumFGE;					// SEG6
            LCDBUF07 |= NumABCD;				// SEG7
            break;
        case 5:									// drawing number 5(use SEG0 and SEG1)
            LCDBUF00 &= NumMask;				// NumMask = 1111 0000
            LCDBUF01 &= NumMask;				// NumMask = 1111 0000
            LCDBUF00 |= NumFGE;					// SEG0
            LCDBUF01 |= NumABCD;				// SEG1
            break;
        case 6:									// drawing number 6(use SEG2 and SEG3)
            LCDBUF02 &= (NumMask|0x08);			// NumMask = 1111 1000 (1111 0000 or 0000 1000) 
            LCDBUF03 &= NumMask;				// NumMask = 1111 0000
            LCDBUF02 |= NumFGE;					// SEG2
            LCDBUF03 |= NumABCD;				// SEG3
            break;
        case 7:									// drawing number 7(use SEG4 and SEG5)
            LCDBUF04 &= (NumMask|0x08);			// NumMask = 1111 1000 (1111 0000 or 0000 1000) 
            LCDBUF05 &= NumMask;				// NumMask = 1111 0000
            LCDBUF04 |= NumFGE;					// SEG4
            LCDBUF05 |= NumABCD;				// SEG5
            break;
        case 8:									// drawing number 8(use SEG12 and SEG13)
            LCDBUF13 &= NumMask;				// NumMask = 1111 0000
            LCDBUF12 &= NumMask;				// NumMask = 1111 0000
            LCDBUF13 |= NumFGE;					// SEG13
            LCDBUF12 |= NumABCD;				// SEG12
            break;
        case 9:									// drawing number 9(use SEG10 and SEG11)
            LCDBUF11 &= NumMask;				// NumMask = 1111 0000
            LCDBUF10 &= NumMask;				// NumMask = 1111 0000
            LCDBUF11 |= NumFGE;					// SEG11
            LCDBUF10 |= NumABCD;				// SEG10
            break;
        case 10:								// drawing number 10useSEG8SEG9
            LCDBUF09 &= NumMask;				// NumMask = 1111 0000
            LCDBUF08 &= NumMask;				// NumMask = 1111 0000
            LCDBUF09 |= NumFGE;					// SEG9
            LCDBUF08 |= NumABCD;				// SEG8
            break;
        default:
        // Add your code here ......
		   break;
    }
}



//*********************************************************************
// name			: Bat_Decode
// brief		: battery level display for GDC03077
// note			: 
//*********************************************************************
void Bat_Decode(uchar Num)
{
    uchar NumBat=0, NumMask;

    NumMask  = 0xf0;							// NumMask  = 1111 0000
    switch(Num){
        case 0:
            NumBat = 0x00;						// all battery display off(S1,S2,S3,S4)  0000 0000
            break;
        case 1:
            NumBat = 0x08;						// display 0% battery level (S1)           0000 1000
            break;
        case 2:
            NumBat = 0x08 | 0x01;				// display 33% battery level (S1,S4)        0000 1001 (0000 1000 or 0000 0001)
            break;
        case 3:
            NumBat = 0x08 | 0x03;				// display 67% battery level (S1,S3,S4)     0000 1011 (0000 1000 or 0000 0011)
            break;
        case 4:
            NumBat = 0x08 | 0x07;				// display 100% battery level (S1,S2,S3,S4) 0000 1111 (0000 1000 or 0000 0111)
            break;
        default:
        // Add your code here ......
        break;
    }
    LCDBUF06 &= NumMask;
    LCDBUF06 |= NumBat;
}


//*********************************************************************
// name			: Percent_Decode
// brief		: percentage display for GDC03077 
// note			: 
//*********************************************************************
void Percent_Decode(uchar Num)
{
    uchar NumPer=0, NumMask;
	
	
	uchar Percent_line;
	uchar Percent_se1;
	uchar Percent_se2;
	uchar Percent_se3;
	uchar Percent_se4;
	
    NumMask  = 0x0f;							// NumMask  = 0000 1111
	
	
	
    switch(Num){
        case 0:								    // percentage displayȫ
			Percent_line = 0x00 << 4;			
			Percent_se1  = 0x00 << 4;
			Percent_se2  = 0x00 << 4;
			Percent_se3  = 0x00 << 4;
			Percent_se4  = 0x00 << 4;

			LCDBUF08  &= NumMask;
			LCDBUF08  |= Percent_line;

			LCDBUF12  &= NumMask;
			LCDBUF12  |= Percent_se1;
			
			LCDBUF11  &= NumMask;
			LCDBUF11  |= Percent_se2;
			
			LCDBUF10  &= NumMask;
			LCDBUF10  |= Percent_se3;
			
			LCDBUF09  &= NumMask;
			LCDBUF09  |= Percent_se4;
            break;
        case 1:									// percentage display (100%)
			Percent_line = 0x08 << 4;			
			Percent_se1  = 0x0F << 4;
			Percent_se2  = 0x0F << 4;
			Percent_se3  = 0x0F << 4;
			Percent_se4  = 0x00 << 4;  
			
			LCDBUF08  &= NumMask;
			LCDBUF08  |= Percent_line;

			LCDBUF12  &= NumMask;
			LCDBUF12  |= Percent_se1;
			
			LCDBUF11  &= NumMask;
			LCDBUF11  |= Percent_se2;
			
			LCDBUF10  &= NumMask;
			LCDBUF10  |= Percent_se3;
			
			LCDBUF09  &= NumMask;
			LCDBUF09  |= Percent_se4;
            break;
        case 2:									// percentage display75
			Percent_line = 0x08 << 4;			
			Percent_se1  = 0x0F << 4;
			Percent_se2  = 0x0F << 4;
			Percent_se3  = 0x00 << 4;
			Percent_se4  = 0x00 << 4;
			
			LCDBUF08  &= NumMask;
			LCDBUF08  |= Percent_line;

			LCDBUF12  &= NumMask;
			LCDBUF12  |= Percent_se1;
			
			LCDBUF11  &= NumMask;
			LCDBUF11  |= Percent_se2;
			
			LCDBUF10  &= NumMask;
			LCDBUF10  |= Percent_se3;
			
			LCDBUF09  &= NumMask;
			LCDBUF09  |= Percent_se4;
            break;
        case 3:									// percentage display
			Percent_line = 0x08 << 4;			
			Percent_se1  = 0x0F << 4;
			Percent_se2  = 0x00 << 4;
			Percent_se3  = 0x00 << 4;
			Percent_se4  = 0x00 << 4;
			
			LCDBUF08  &= NumMask;
			LCDBUF08  |= Percent_line;

			LCDBUF12  &= NumMask;
			LCDBUF12  |= Percent_se1;
			
			LCDBUF11  &= NumMask;
			LCDBUF11  |= Percent_se2;
			
			LCDBUF10  &= NumMask;
			LCDBUF10  |= Percent_se3;
			
			LCDBUF09  &= NumMask;
			LCDBUF09  |= Percent_se4;
            break;
        case 4:									// percentage display
			Percent_line = 0x08 << 4;			
			Percent_se1  = 0x0C << 4;
			Percent_se2  = 0x00 << 4;
			Percent_se3  = 0x00 << 4;
			Percent_se4  = 0x00 << 4;
			
			LCDBUF08  &= NumMask;
			LCDBUF08  |= Percent_line;

			LCDBUF12  &= NumMask;
			LCDBUF12  |= Percent_se1;
			
			LCDBUF11  &= NumMask;
			LCDBUF11  |= Percent_se2;
			
			LCDBUF10  &= NumMask;
			LCDBUF10  |= Percent_se3;
			
			LCDBUF09  &= NumMask;
			LCDBUF09  |= Percent_se4;
            break;
        default:
        // Add your code here ......
        break;
    }
	



}




