//******************************************
// file			:
// version		:
// brief		:
// note			:
//******************************************
#include "main.h"

User_I2C  Flow;
extern uchar wBuf[16];
//*********************************************************************
// file			: I2C_SetSlaveAddr()
// brief		: set slave address
// note			:
//*********************************************************************
void I2C_SetSlaveAddr(uchar addr)
{
    I2C0AR = addr & 0xfe;               // SA0[7:1]: Slave address setting
                                        // &0xfe   : and clear bit0
}


//*********************************************************************
// file			: I2C_Init
// brief		: enable I2C and start I2C
// note			:
//*********************************************************************
void I2C_Init(void)
{
    P2CR_P3  = 0;
    P2CR_P4  = 0;
    POFFCR1_SBI0EN  = 1;                // SBI0 Enable
    while( P2PRD_P3==0 || P2PRD_P4==0 ){ CLR_WDT; } // wait for high level
    P2DR_P3    = 0;
    P2DR_P4    = 0;
    P2FC_P3    = 1;
    P2FC_P4    = 1;
    P2OUTCR_P3 = 1;
    P2OUTCR_P4 = 1;
    P2CR_P3    = 1;
    P2CR_P4    = 1;

//----------------------------------//
    SBI0CR2 = 0x18;
    SBI0CR1 = 0x10;

}

//*********************************************************************
// file			: I2C_IntEnable
// brief		: enable I2C interrupt
// note			: after execution, Interrupt Master Allowed Flag (IMF) will be enabled
//*********************************************************************
void I2C_IntEnable(void)
{
    DI;
    IL_INTSBI = 0;
    EI_INTSBI = 1;
    EI;
}


//*********************************************************************
// file			: I2C_IntDisable 
// brief		: disable I2C interrupt
// note			: after execution, Interrupt Master Allowed Flag (IMF) will be enabled
//*********************************************************************
void I2C_IntDisable(void)
{
    DI;
    EI_INTSBI = 0;
    IL_INTSBI = 0;  
    EI;
}


//*********************************************************************
// file			: I2C_Start 
// brief		: generate start condition
// note			:
//*********************************************************************
void I2C_Start(uchar data, uchar address)
{
    uchar tWait = 0;

    while( SBI0SR2_BB == 1 ){}          // SBI0SR2.BB: confirms bus is free

    SBI0CR1 = 0x10;                     // SBI0CR1 needs to be set for each I2C_Start.
//****** Write data to the registers before the start condition is generated ******
    SBI0DBR = data;                     // set slave Device ID
    SBI0CR2 = 0xf8;                     // Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
    while( SBI0SR2_PIN == 1 ){}        // wait for releasing interrupt service request    
    while( SBI0SR2_LRB == 1 ){         // wait for last received bit is "0"                                   
        tWait++;                        // if no response, SCK is always low level
        if( tWait>250 ){                // jump out of loop if no response for too long       
            I2C_Stop();                 // I2C stop -> release SCK
            break;
        }
    }

//... set slave address..................
    tWait = 0;
    SBI0DBR = address;                  // set slave address
    while( SBI0SR2_PIN == 1 ){}         // wait for releasing interrupt service request
    while( SBI0SR2_LRB == 1 ){          // wait for last received bit is "0"                                       
        tWait++;                        // if no response, SCK is always low level
        if( tWait>250 ){                // jump out of loop if no response for too long       
            I2C_Stop();                 // I2C stop -> release SCK
            break;
        }
    }
}


//*********************************************************************
// file			: I2C_Restart 
// brief		: generate restart condition
// note			:
//*********************************************************************
void I2C_Restart(uchar data)
{
    uchar tWait = 0;

    SBI0CR2 = 0x18;
    while( SBI0SR2_BB == 1 ){}          // SBI0SR2.BB: confirms bus is free
    while( SBI0SR2_LRB == 0 ){}         // wait <LRB>=1**********************

//****** Write data to the registers before the start condition is generated ******
    SBI0DBR = data;
    SBI0CR2 = 0xf8;                     // Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
    while( SBI0SR2_PIN == 1 ){}         // wait for releasing interrupt service request
    while( SBI0SR2_LRB == 1 ){          // wait for last received bit is "0"                                       
        tWait++;                        // if no response, SCK is always low level
        if( tWait>250 ){                // jump out of loop if no response for too long       
            I2C_Stop();                 // I2C stop -> release SCK
            break;
        }
    }
}


//*********************************************************************
// file			: I2C_Stop 
// brief		: generate stop condition
// note			:
//*********************************************************************
void I2C_Stop(void)
{
    SBI0CR2 = 0xd8;                     // Set SBI0CR2<MST>, <TRX> and <PIN> to "1" and SBI0CR2<BB> to "0"
    while( SBI0SR2_BB == 1 ){}          // SBI0SR2.BB: confirms bus is free
}


//*********************************************************************
// file			: I2C_ByteWrite 
// brief		: byte write * len
// note			:
//*********************************************************************
void I2C_ByteWrite(uchar *data, uchar len)
{
    uchar i, tWait;

    for( i=0; i<len; i++ ){ 
        tWait = 0;
        SBI0DBR = *data;
        data++;
        while( SBI0SR2_PIN == 1 ){}     // wait for releasing interrupt service request
        while( SBI0SR2_LRB == 1 ){      // wait for last received bit is "0"                                       // if no response, SCK is always low level
            tWait++;
            if( tWait>250 ){            // jump out of loop if no response for too long           
                I2C_Stop();             // I2C stop -> release SCK
                break;
            }
        }
    }

}


//*********************************************************************
// file			: I2C_ByteRead 
// brief		: byte read * len
// note			:
//*********************************************************************
void I2C_ByteRead(uchar *data, uchar len)
{
    uchar i;

    len -= 1;

    for( i=0; i<len; i++ ){
        SBI0DBR   = 0x00;               // write dummy data to set PIN = 1
        while( SBI0SR2_PIN == 1 ){}     // wait for releasing interrupt service request
        *data = SBI0DBR;
        data++;
    }


    SBI0CR1_ACK = 0;                    // ACK=0
    SBI0DBR   = 0x00;                   // write dummy data to set PIN = 1
    while( SBI0SR2_PIN == 1 ){}         // wait for releasing interrupt service request
    *data = SBI0DBR;

//  Termination of data transfer by master mode
//  Sent 1 bit negative acknowledge............
    SBI0CR1_BC = 1;                     // BC, bit count = 1
    SBI0DBR   = 0x00;                   // write dummy data to set PIN = 1
    while( SBI0SR2_PIN == 1 ){}         // wait for releasing interrupt service request
}



//*********************************************************************
// file			: IntSBI
// brief		: I2C interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntSBI(void)
{
    BIT_SBI0SR2 I2C_Flag;
    
    *((uchar *)(&I2C_Flag)) = SBI0SR2;
    
//----------------------------------------------------------//
//****** Receiver at slave mode ****************************//
//----------------------------------------------------------//
    if( I2C_Flag.TRX == 0 ){                            // TRX: Transmitter/Receiver selection status monitor.==> 0).Receiver, 1).Transmitter   
        if( I2C_Flag.AAS == 1 ){                        // AAS: Slave address match detection monitor       
            Flow.sFlag.match_addr = 1;                  // first address match detection
            Flow.iByte            = 0;
            Flow.sFlag.R1W0       = I2C_Flag.LRB;       // 1:read, 0:write.
            SBI0DBR               = 0;                  // Write the dummy data (0x00) to the SBI0DBR to set SBI0CR2<PIN> to "1"

        }
        else{       
            if( Flow.sFlag.match_addr == 1 ){           // get address first            
                Flow.iAddr_st             = SBI0DBR;
                Flow.iAddr_pt             = Flow.iAddr_st;
                Flow.iByte                = Flow.iAddr_st;
                Flow.sFlag.match_addr = 0;
                SBI0DBR              = 0;               // Write the dummy data (0x00) to the SBI0DBR to set SBI0CR2<PIN> to "1"
            }
            else{           
                wBuf[Flow.iByte] = SBI0DBR;             // get write data from master
                Flow.iByte++;
                Flow.iByte &= 0x0f;                     // max. 16 byte: then overwrite data at 1st has get
                SBI0DBR    = 0;                         // Write the dummy data (0x00) to the SBI0DBR to set SBI0CR2<PIN> to "1"
            }
        }
    }

//----------------------------------------------------------//
//****** Transmitter at slave mode *************************//
//----------------------------------------------------------//
    else
    {
        if( I2C_Flag.AAS == 1 ){        // AAS: Slave address match detection monitor       
            SBI0DBR = wBuf[Flow.iByte];
            Flow.iByte++;
        }
        else{
            SBI0DBR = wBuf[Flow.iByte];
            Flow.iByte++;

        //----------------------------------------------------------//
        //****** Master non-ACK for last byte **********************//
        //----------------------------------------------------------//
            if( I2C_Flag.LRB == 1 ){    // LRB: Last received bit monitor           
            }
            else{           
            }
        }
    }   
}






